import gradio as gr
import whisper
from openai import OpenAI
from dotenv import load_dotenv

load_dotenv()

starting_prompt = """Jesteś asystentem.
Możesz rozmawiać z użytkownikiem i obsługiwać e-maila. E-maile muszą mieć temat, odbiorcę i treść.
Otrzymasz albo instrukcje zaczynające się od [Instrukcja] albo dane wejściowe od użytkownika [Użytkownik].Wykonaj zadanie zgodnie z instrukcjami.
"""

prompts = {
           "START": "Określ intencję wyrażoną we wprowadzonych danych.Odpowiedz jednym słowem: NAPISZ_EMAIL, PYTANIE, INNE.",
           "PYTANIE": "Odpowiedz jednym słowem: ODPOWIEDŹ, jeżeli możesz odpowiedzieć na pytanie, WIĘCEJ, jeżeli potrzebujesz więcej informacji, lub INNE, jeżeli nie możesz odpowiedzieć.",
           "ODPOWIEDŹ": "Odpowiedz na pytanie.",
           "WIĘCEJ": "Poproś o więcej informacji.",
           "INNE": "Odpowiedz, że nie możesz odpowiedzieć na pytanie lub wykonać polecenia.",
           "NAPISZ_EMAIL": 'Odpowiedz "WIĘCEJ", jeżeli brakuje tematu, adresu odbiorcy lub treści. Jeżeli masz wszystkie informacje, odpowiedz \
           "AKCJA_NAPISZ_EMAIL | temat:temat, odbiorca:odbiorca, treść:treść".',
}

actions = ['ACTION_WRITE_EMAIL']


class Discussion:
    """
    Klasa do rozmowy z asystentem rozmowy.

    Parametry:
        state (str): Bieżący status rozmowy.
        messages_history (list): Lista słowników pokazująca historię wiadomości wymienionych podczas rozmowy.
        client: Instancja klienta OpenAI.
        stt_model: Model przetwarzający mowę na tekst, który służy do tworzenia transkrypcji.

    Metody:
        generate_answer: Generowanie odpowiedzi w oparciu o podane wiadomości.
        reset: Resetuje rozmowę, a w rezultacie rozmowa wraca do stanu początkowego.
        do_action: Wykonuje określone działanie.
        transcribe: Utworzenie transkrypcji dla danego pliku audio.
        discuss_from_audio: Rozpoczęcie rozmowy na podstawie transkrypcji z pliku audio.
        discuss: Dalszy ciąg rozmowy w oparciu o podane dane wejściowe.
    """

    def __init__(
            self, state='START',
            messages_history=[{'role': 'user',
                               'content': f'{starting_prompt}'}]) -> None:
        self.state = state
        self.messages_history = messages_history
        self.client = OpenAI()
        self.stt_model = whisper.load_model("base")
        pass

    def generate_answer(self, messages):
        response = self.client.chat.completions.create(
            model="gpt-4-turbo",
            messages=messages)
        return (response.choices[0].message.content)

    def reset(self, start_state='START'):
        self.messages_history = [
            {'role': 'user', 'content': f'{starting_prompt}'}]
        self.state = start_state
        self.previous_state = None

    def reset_to_previous_state(self):
        self.state = self.previous_state
        self.previous_state = None

    def to_state(self, state):
        self.previous_state = self.state
        self.state = state

    def do_action(self, action):
        """
        Wykonanie określoneego działania.

        Args:
            action (str): Akcja do wykonania.
        """
        print(f'Usuwanie błędów dla={action}')
        pass

    def transcribe(self, file):
        transcription = self.stt_model.transcribe(file)
        return transcription['text']

    def discuss_from_audio(self, file):
        if file:
            # Utwórz transkrypcję pliku audio i użyj uzyskany tekst do dalszej rozmowy.
            return self.discuss(f'[User] {self.transcribe(file)}')
        # # Wyczyść dane, jeśli nie ma pliku.
        return ''

    def discuss(self, input=None):
        if input is not None:
            self.messages_history.append({"role": "user", "content": input})

        # Generowanie odpowiedzi
        completion = self.generate_answer(
            self.messages_history +
            [{"role": "user", "content": prompts[self.state]}])

        # Czy uzupełnienie wymaga akcji ?
        if completion.split("|")[0].strip() in actions:
            action = completion.split("|")[0].strip()
            self.to_state(action)
            self.do_action(completion)
            # Kontynuacja rozmowy.
            return self.discuss()
            # Czy uzupełnienia są nowym stanem?
        elif completion in prompts:
            self.to_state(completion)
            # Kontynuacja rozmowy
            return self.discuss()
        # Czy uzupełnienie to dane wyjściowe dla użytkownika?
        else:
            self.messages_history.append(
                {"role": "assistant", "content": completion})
            if self.state != 'MORE':
                # Powrót do początku
                self.reset()
            else:
                # Powrót do poprzedniego stanu
                self.reset_to_previous_state()
            return completion


if __name__ == '__main__':
    discussion = Discussion()

    gr.Interface(
        theme=gr.themes.Soft(),
        fn=discussion.discuss_from_audio,
        live=True,
        inputs=gr.Audio(sources="microphone", type="filepath"),
        outputs="text").launch()

    # Jeśli chcesz używać wiersza polenia, zamiast Gradio, usuń kod widoczny powyżej i użyj tego:
    # while True:
    #     message = input('Użytkownik: ')
    #     print(f'Asystent: {discussion.discuss(message)}')